import argparse
import json
import os
import sys
import subprocess
import urllib.request
import zipfile
import shutil
import io
from datetime import datetime

if sys.platform == "win32":
    # Force UTF-8 for standard output to handle emojis
    sys.stdout = io.TextIOWrapper(sys.stdout.buffer, encoding='utf-8')
    sys.stderr = io.TextIOWrapper(sys.stderr.buffer, encoding='utf-8')

def log(message):
    print(f"[*] {message}")

def error(message):
    print(f"[!] Error: {message}")
    sys.exit(1)

def handle_request_error(e, stage="Validation"):
    emsg = str(e)
    if "[SSL: CERTIFICATE_VERIFY_FAILED]" in emsg and sys.platform == "darwin":
        v = sys.version_info
        py_folder = f"Python {v.major}.{v.minor}"
        cert_cmd = f"/Applications/{py_folder}/Install Certificates.command"
        
        log("Detected SSL Certificate issue. Attempting self-correction...")
        
        if os.path.exists(cert_cmd):
            try:
                log(f"Running certificate installer: {cert_cmd}")
                subprocess.run(["sh", cert_cmd], check=True, capture_output=True)
                log("Certificates installed successfully! Retrying...")
                return "retry"
            except Exception as se:
                log(f"Auto-install failed: {str(se)}")
        
        log("Applying temporary SSL bypass to complete activation...")
        import ssl
        ssl._create_default_https_context = ssl._create_unverified_context
        return "retry"

    error(f"{stage} failed: {emsg}")

def cleanup_metadata(path="."):
    """Removes __MACOSX folders and ._* files recursively."""
    log("Cleaning up metadata files...")
    for root, dirs, files in os.walk(path, topdown=False):
        for d in dirs:
            if d == "__MACOSX":
                full_path = os.path.join(root, d)
                try:
                    shutil.rmtree(full_path)
                except Exception:
                    pass
        for f in files:
            if f.startswith("._"):
                full_path = os.path.join(root, f)
                try:
                    os.remove(full_path)
                except Exception:
                    pass

def main():
    parser = argparse.ArgumentParser(description="CC Activation Skill Logic")
    parser.add_argument("--key", required=True, help="CCOS License Key")
    parser.add_argument("--email", required=True, help="Email Address")
    parser.add_argument("--os", default="Mac", help="Operating System Type")
    args = parser.parse_args()

    api_key = "sb_publishable_HkQ9umD_WrZn7fXwr1KCxw_WumM64kf"
    base_url = "https://dwgqsttkmibrdaizpkzp.supabase.co"
    

    # 2. Validate & Fetch Assets
    log("Validating license and fetching assets...")
    assets_url = f"{base_url}/functions/v1/ccos-deliver-assets"
    data = json.dumps({"license_key": args.key, "email": args.email}).encode("utf-8")
    
    while True:
        try:
            req = urllib.request.Request(
                assets_url, 
                data=data, 
                headers={
                    "apikey": api_key,
                    "Authorization": f"Bearer {api_key}",
                    "Content-Type": "application/json"
                },
                method="POST"
            )
            with urllib.request.urlopen(req) as response:
                res_data = json.loads(response.read().decode("utf-8"))
            break
        except Exception as e:
            if handle_request_error(e, "Validation") == "retry":
                continue
            break

    if "signed_url" not in res_data:
        error("Validation failed: No signed URL returned. Check your key and email.")

    signed_url = res_data["signed_url"]
    profile_data = res_data.get("profile", {})
    user_name = profile_data.get("name", "Explorer")
    
    # 3. Download Assets
    log("Downloading CCOS intelligence pack...")
    temp_zip = "temp_ccos.zip"
    while True:
        try:
            urllib.request.urlretrieve(signed_url, temp_zip)
            break
        except Exception as e:
            if handle_request_error(e, "Download") == "retry":
                continue
            break

    # 4. Unzip Assets
    log("Installing assets...")
    try:
        with zipfile.ZipFile(temp_zip, 'r') as zip_ref:
            zip_ref.extractall(".")
        os.remove(temp_zip)
        
        # Cleanup
        cleanup_metadata(".")
    except Exception as e:
        error(f"Installation failed: {str(e)}")

    # 5. Local Identity Installation
    log("Updating identity file...")
    identity_path = "my-ccos/identity.md"
    os.makedirs(os.path.dirname(identity_path), exist_ok=True)
    
    identity_content = f"""# User Identity: {user_name}

- **Email**: {args.email}
- **License Key**: {args.key}
- **Status**: Verified ✅
- **Sync Date**: {datetime.now().strftime('%Y-%m-%d')}

## Strategic Context
- **Niche**: {profile_data.get('niche', 'Not Set')}
- **Goals**: {profile_data.get('goal', 'Not Set')}
- **Persona**: {profile_data.get('persona', 'Not Set')}
"""
    with open(identity_path, "w", encoding="utf-8") as f:
        f.write(identity_content)

    # 6. Usage Stats Check-in
    log("Finalizing activation...")
    stats_url = f"{base_url}/rest/v1/usage_stats"
    stats_data = json.dumps({
        "user_id": profile_data.get("user_id", "unknown"),
        "platform": "Desktop",
        "os": args.os
    }).encode("utf-8")
    
    try:
        req = urllib.request.Request(
            stats_url,
            data=stats_data,
            headers={
                "apikey": api_key,
                "Authorization": f"Bearer {api_key}",
                "Content-Type": "application/json"
            },
            method="POST"
        )
        with urllib.request.urlopen(req) as response:
            pass
    except Exception:
        log("Warning: Could not log usage stats, but activation was successful.")

    print(json.dumps({
        "status": "success",
        "name": user_name,
        "identity_path": identity_path
    }))

if __name__ == "__main__":
    main()
