import argparse
import json
import os
import sys
import urllib.request
import zipfile
import shutil
from datetime import datetime
import io

if sys.platform == "win32":
    # Force UTF-8 for standard output to handle emojis
    sys.stdout = io.TextIOWrapper(sys.stdout.buffer, encoding='utf-8')
    sys.stderr = io.TextIOWrapper(sys.stderr.buffer, encoding='utf-8')

def log(message):
    print(f"[*] {message}")

def error(message):
    print(f"[!] Error: {message}")
    sys.exit(1)

def cleanup_metadata(path="."):
    """Removes __MACOSX folders and ._* files recursively."""
    for root, dirs, files in os.walk(path, topdown=False):
        for d in dirs:
            if d == "__MACOSX":
                full_path = os.path.join(root, d)
                try:
                    shutil.rmtree(full_path)
                except Exception:
                    pass
        for f in files:
            if f.startswith("._"):
                full_path = os.path.join(root, f)
                try:
                    os.remove(full_path)
                except Exception:
                    pass

def get_current_version(changelog_path="ccos/system/releases/CHANGELOG.md"):
    """Extracts the latest version from the CHANGELOG.md file."""
    if not os.path.exists(changelog_path):
        # Fallback to identity file if changelog is missing (backward compatibility)
        return get_version_from_identity()
    
    try:
        with open(changelog_path, "r", encoding="utf-8") as f:
            for line in f:
                # Look for lines starting with '## [' which denote version headers
                # Example: ## [1.0.1] - 2026-01-12
                if line.strip().startswith("## ["):
                    # Extract content between first '[' and ']'
                    start = line.find("[") + 1
                    end = line.find("]")
                    if start > 0 and end > start:
                        return line[start:end].strip()
    except Exception:
        pass
    
    return "0.0.0"

def get_version_from_identity(identity_path="my-ccos/identity.md"):
    """Fallback: Extracts version from identity.md"""
    if not os.path.exists(identity_path):
        return "0.0.0"
    
    try:
        with open(identity_path, "r", encoding="utf-8") as f:
            for line in f:
                if "Release" in line and ":" in line:
                    parts = line.split(":", 1)
                    if "Release" in parts[0]:
                        return parts[1].strip()
    except Exception:
        pass
    return "0.0.0"

def get_identity_data(identity_path="my-ccos/identity.md"):
    data = {"email": "", "key": "", "name": ""}
    if not os.path.exists(identity_path):
        return data
    
    with open(identity_path, "r", encoding="utf-8") as f:
        content = f.read()
        if "User Identity:" in content:
            data["name"] = content.split("User Identity:")[1].split("\n")[0].strip()
        if "**Email**:" in content:
            data["email"] = content.split("**Email**:")[1].split("\n")[0].strip()
        if "**License Key**:" in content:
            data["key"] = content.split("**License Key**:")[1].split("\n")[0].strip()
    return data

def main():
    parser = argparse.ArgumentParser(description="CC Update Skill Logic")
    parser.add_argument("--os", default="Mac", help="Operating System Type")
    args = parser.parse_args()

    api_key = "sb_publishable_HkQ9umD_WrZn7fXwr1KCxw_WumM64kf"
    base_url = "https://dwgqsttkmibrdaizpkzp.supabase.co"
    identity_path = "my-ccos/identity.md"

    identity = get_identity_data(identity_path)
    if not identity["key"] or not identity["email"]:
        error("No active license found. Please activate your license first.")

    current_version = get_current_version()
    log(f"Current version: {current_version}")

    # 1. Check for updates
    log("Checking for updates...")
    update_url = f"{base_url}/functions/v1/ccos-check-updates"
    data = json.dumps({
        "license_key": identity["key"],
        "email": identity["email"],
        "current_version": current_version
    }).encode("utf-8")
    
    try:
        req = urllib.request.Request(
            update_url, 
            data=data, 
            headers={
                "apikey": api_key,
                "Authorization": f"Bearer {api_key}",
                "Content-Type": "application/json"
            },
            method="POST"
        )
        with urllib.request.urlopen(req) as response:
            res_data = json.loads(response.read().decode("utf-8"))
    except Exception as e:
        error(f"Update check failed: {str(e)}")

    if not res_data.get("update_available"):
        log("You are already on the latest version!")
        print(json.dumps({"status": "no_update", "version": current_version}))
        return

    latest_version = res_data["latest_version"]
    log(f"Update available! New version: {latest_version}")
    
    # 2. Download Assets
    log("Downloading update...")
    signed_url = res_data["signed_url"]
    temp_zip = "temp_update.zip"
    try:
        urllib.request.urlretrieve(signed_url, temp_zip)
    except Exception as e:
        error(f"Download failed: {str(e)}")

    # 3. Unzip Assets
    log("Installing update...")
    try:
        with zipfile.ZipFile(temp_zip, 'r') as zip_ref:
            zip_ref.extractall(".")
        os.remove(temp_zip)
        cleanup_metadata(".")
    except Exception as e:
        error(f"Installation failed: {str(e)}")

    # 4. Update Identity Sync Date
    log("Updating identity sync date...")
    try:
        with open(identity_path, "r", encoding="utf-8") as f:
            lines = f.readlines()
        
        new_lines = []
        for line in lines:
            if "Sync Date:" in line:
                new_lines.append(f"- **Sync Date**: {datetime.now().strftime('%Y-%m-%d')}\n")
            elif "Release:" in line:
                continue # Skip old release lines in identity.md
            else:
                new_lines.append(line)
        
        with open(identity_path, "w", encoding="utf-8") as f:
            f.writelines(new_lines)
    except Exception as e:
        log(f"Warning: Could not update identity file: {str(e)}")

    print(json.dumps({
        "status": "success",
        "old_version": current_version,
        "new_version": latest_version
    }))

if __name__ == "__main__":
    main()
